//
//  TodoItemDetail.swift
//  Do It
//
//  Created by Jim Dovey on 9/9/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI
import CoreData

struct TodoItemDetail: View {
    // START:NewState
    @ObservedObject var item: TodoItem
    @Environment(\.managedObjectContext) var objectContext

    @State var showingEditor = false
    // END:NewState
    
    var gradient: LinearGradient {
        LinearGradient(
            gradient: Gradient(colors: [
                Color.black.opacity(0.6),
                Color.black.opacity(0),
            ]),
            startPoint: .bottom,
            endPoint: .init(x: 0.5, y: 0.1)
        )
    }

    var headerBackground: some View {
        Rectangle()
            .fill(item.list?.color.uiColor ?? .blue)
            .overlay(gradient)
            .edgesIgnoringSafeArea(.top)
    }

    // START:EditorView
    var editor: some View {
        let context = self.objectContext.editingContext()//<label id="code.7.detail.editor.view.context"/>
        guard let editItem = context.realize(self.item) else {//<label id="code.7.detail.editor.view.realize"/>
            preconditionFailure("Failed to get edit version of existing item")
        }
        return NavigationView {
            TodoItemEditor(item: editItem)
                .environment(\.managedObjectContext, context)//<label id="code.7.detail.edit.environment"/>
        }
    }
    // END:EditorView

    var editorButton: some View {
        Button(action: {
            self.showingEditor.toggle()
        }) {
            Image(systemName: "square.and.pencil")
                .imageScale(.large)
                .foregroundColor(.accentColor)
        }
        .accessibility(label: Text("Edit"))
    }

    var body: some View {
        ScrollView(.vertical) {
            VStack(alignment: .leading) {
                VStack(alignment: .leading, spacing: 8) {
                    Text(item.title ?? "New Item")
                        .font(.title)
                        .bold()
                        .layoutPriority(1)
                    
                    HStack(alignment: .firstTextBaseline) {
                        Text("Priority: ") +
                            Text(item.priority.localizedName).bold()
                        Spacer()
                        if item.date != nil {
                            Text("\(item.date!, formatter: Formatters.mediumDate)")
                        } else {
                            Text("No Due Date")
                        }
                    }
                    .font(.subheadline)
                }
                .foregroundColor(.white)
                .padding()
                .padding(.top)
                    .background(headerBackground)
                    .layoutPriority(1)
                
                if item.notes != nil {
                    Text(verbatim: self.item.notes!)
                        .padding()
                }
                Spacer()
            }
        }
        .navigationBarTitle("", displayMode: .inline)
        .navigationBarItems(trailing: editorButton)
        // START:BodyUpdates
        .sheet(isPresented: $showingEditor, onDismiss: {
            if self.item.hasPersistentChangedValues {
                UIApplication.shared.saveContext()
            }
        }, content: { self.editor })
        // END:BodyUpdates
    }
}

struct TodoItemDetail_Previews: PreviewProvider {
    static var previews: some View {
        single
            .environment(\.managedObjectContext, PreviewDataStore.shared.viewContext)
    }
    
    static var single: some View {
        NavigationView {
            TodoItemDetail(item: PreviewDataStore.shared.sampleItem)
        }
    }
    
    static var allColorSchemePreviews: some View {
        ForEach(ColorScheme.allCases, id: \.self) { scheme in
            NavigationView {
                TodoItemDetail(item: PreviewDataStore.shared.sampleItem)
            }
            .colorScheme(scheme)
            .previewDisplayName(String(describing: scheme))
        }
    }
    
    static var multiDevicePreviews: some View {
        ForEach(["iPhone 11", "iPhone SE"], id: \.self) { name in
            TodoItemDetail(item: PreviewDataStore.shared.sampleItem)
                .previewDevice(PreviewDevice(rawValue: name))
                .previewDisplayName(name)
        }
    }
    
    static var allLocalizations: some View {
        let localePairs: [(Locale, LayoutDirection)] = [
            (Locale(identifier: "en-US"), .leftToRight),
            (Locale(identifier: "pl"), .leftToRight),
            (Locale(identifier: "ar"), .rightToLeft),
        ]
        
        return ForEach(localePairs, id: \.self.0) { value in
            TodoItemDetail(item: PreviewDataStore.shared.sampleItem)
                .environment(\.locale, value.0)
                .environment(\.layoutDirection, value.1)
                .previewDisplayName(value.0.languageCode ?? value.0.identifier)
        }
    }
    
    static var allSizeCategories: some View {
        ForEach(ContentSizeCategory.allCases, id: \.self) { category in
            TodoItemDetail(item: PreviewDataStore.shared.sampleItem)
                .environment(\.sizeCategory, category)
                .previewDisplayName(String(describing: category))
        }
    }
}
